/**
 * \file keystore.c
 *
 * \brief Functions to manage the key-store
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "keystore.h"
#include "daemon_log.h"
#include "helper.h"

static const char *extensions[KEYSTORE_END] = {
    NULL,
    ".pending"
};

static void keystore_ctx_free_dir(keystore_ctx_t *ctx)
{
    if (ctx->keystore_dir_fd >= 0) {
        close(ctx->keystore_dir_fd);
    }
    ctx->keystore_dir_fd = -1;
}

static sdc_error_t keystore_commit_pending(keystore_ctx_t *ctx, bool initial);

/**
 * Initialize the directory fd for accessing the keystore
 *
 * \param ctx       Keystore context
 * \param keystore_path  Path to the keystore basename (including filename)
 * \return corresponding sdc_error
 */
static sdc_error_t keystore_ctx_init_dir(keystore_ctx_t *ctx,
                                         const char *keystore_path)
{
    /* we need to do this very complicated approach as
     * dirname might alter the string
     * and might return a pointer to this string or
     * to a statically allocated string (e.g. .)
     */
    char *path_copy;
    char *dir;
    sdc_error_t err;

    ctx->keystore_dir_fd = -1;

    path_copy = strdup(keystore_path);
    if (!path_copy) {
        return SDC_NO_MEM;
    }

    dir = dirname(path_copy);

    err = sdc_helper_create_absolute_dir(dir, true, true);

    if (err == SDC_OK)
        ctx->keystore_dir_fd = open(dir, O_RDONLY | O_DIRECTORY);

    free(path_copy);

    if (ctx->keystore_dir_fd < 0) {
        daemon_log(DAEMON_LOG_ERROR,"Failed to open key_store directory - error: %s\n", strerror(errno));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    return SDC_OK;
}


/**
 * Free the key store names
 *
 * \param ctx       Keystore context
 */
static void keystore_ctx_free_names(keystore_ctx_t *ctx)
{
    int i;

    for (i = 0; i < KEYSTORE_END; i++) {
        free(ctx->keystore_names[i]);
        ctx->keystore_names[i] = NULL;
    }
}


/**
 * Initialize the key store names
 *
 * \param ctx       Keystore context
 * \param keystore_path  Path to the keystore basename (including directory)
 * \return corresponding sdc_error
 */
static sdc_error_t keystore_ctx_init_names(keystore_ctx_t *ctx,
                                           const char *keystore_path)
{
    size_t base_len;
    size_t len;
    int i;
    char *path_copy;
    char *keystore_name;
    sdc_error_t err = SDC_OK;

    /* make sure names are initialized */
    for (i = 0; i < KEYSTORE_END; i++) {
        ctx->keystore_names[i] = NULL;
    }

    path_copy = strdup(keystore_path);
    if (!path_copy) {
        return SDC_NO_MEM;
    }

    keystore_name = basename(path_copy);
    base_len = strlen(keystore_name);

    for (i = 0; i < KEYSTORE_END; i++) {
        if (extensions[i]) {
            len = base_len + strlen(extensions[i]) + 1;
            ctx->keystore_names[i] = malloc(len);
            if (!ctx->keystore_names[i]) {
                err = SDC_NO_MEM;
                break;
            }
            snprintf(ctx->keystore_names[i], len, "%s%s",
                     keystore_name, extensions[i]);
        } else {
            ctx->keystore_names[i] = strdup(keystore_name);
            if (!ctx->keystore_names[i]) {
                err = SDC_NO_MEM;
                break;
            }
        }
    }

    free(path_copy);

    if (err != SDC_OK) {
        keystore_ctx_free_names(ctx);
    }

    return err;
}


sdc_error_t keystore_ctx_init(keystore_ctx_t *ctx, const char *keystore_path)
{
    sdc_error_t err;

    err = keystore_ctx_init_dir(ctx, keystore_path);

    if (err == SDC_OK) {
        err = keystore_ctx_init_names(ctx, keystore_path);
        if (err != SDC_OK) {
            keystore_ctx_free_dir(ctx);
        }
    }

    if (err == SDC_OK) {
        err = keyman_ctx_alloc(&ctx->libkeyman_ctx);

        if (err != SDC_OK) {
            keystore_ctx_free_dir(ctx);
            keystore_ctx_free_names(ctx);
        }
    }

    return err;
}

sdc_error_t keystore_ctx_fini(keystore_ctx_t *ctx)
{
    keystore_ctx_free_dir(ctx);
    keystore_ctx_free_names(ctx);
    keyman_ctx_free(ctx->libkeyman_ctx);

    return SDC_OK;
}

sdc_error_t keystore_initialize(keystore_ctx_t *ctx)
{
    sdc_error_t err;
    int errno_err;
    int fd;

    err = keyman_lock_unlock_keystore(ctx->libkeyman_ctx, LIBKEYMAN_KEYSTORE_ALL, true);

    //TODO : Need to implement key store lock support for imx6
    if((err != SDC_OK) && (err != SDC_NOT_SUPPORTED)) /* imx6 not support key store lock */
        return err;

    if (0 == faccessat(ctx->keystore_dir_fd,
                       ctx->keystore_names[KEYSTORE_STANDARD],
                       F_OK, 0)) {
        return SDC_KEY_STORAGE_ALREADY_PRESENT;
    }
    errno_err = errno;
    if (errno_err != ENOENT) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Unexpected error when accessing key-store %s: %s",
                   ctx->keystore_names[KEYSTORE_STANDARD],
                   strerror(errno_err));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }
    if (0 == faccessat(ctx->keystore_dir_fd,
                       ctx->keystore_names[KEYSTORE_PENDING],
                       F_OK, 0)) {
        return SDC_KEY_STORAGE_ALREADY_PRESENT;
    }
    errno_err = errno;
    if (errno_err != ENOENT) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Unexpected error when accessing key-store %s: %s",
                   ctx->keystore_names[KEYSTORE_PENDING],
                   strerror(errno_err));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    /* create new keystore */
    err = SDC_OK;
    fd = openat(ctx->keystore_dir_fd,
                ctx->keystore_names[KEYSTORE_PENDING], O_WRONLY | O_CREAT | O_EXCL, NEW_KEYSTORE_MODE);
    if (fd < 0) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to initialize key-storage - %s",
                   strerror(errno));
        err = SDC_KEY_STORAGE_ACCESS_FAILED;
    } else {
        err = keyman_empty_persistent_keystore(ctx->libkeyman_ctx, fd);

        if (err == SDC_OK) {
            err = keystore_commit_pending(ctx, true);
        }
    }

    return err;
}

sdc_error_t keystore_verify_credentials(keystore_ctx_t *ctx)
{
    sdc_error_t err = SDC_OK;
    struct stat keystore_dir_stat;

    bool key_store_present = false;

    /* Check that the daemon has read, write and execute permissions to the
     * directory (via the user)
     * Check that group has no write permissions
     * Check that others don't have any permissions
     *
     * Note : We don't compare the uid to the current user - we use
     * a dedicated access call to verify (daemon might be started as root)
     */
    if (fstat(ctx->keystore_dir_fd, &keystore_dir_stat)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Fstat on key-store directory failed");
        err = SDC_KEY_STORAGE_ACCESS_FAILED;
    }
    if (err == SDC_OK) {
        if ((keystore_dir_stat.st_mode & S_IRWXU) != S_IRWXU) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Invalid mode (0%o) of key-store directory. User does require rwx permissions.",
                       keystore_dir_stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID));
            err = SDC_KEY_STORAGE_ACCESS_FAILED;
        }
        if (keystore_dir_stat.st_mode & S_IWGRP) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Invalid mode (0%o) of key-store directory. Group must not have w permissions.",
                       keystore_dir_stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID));
            err = SDC_KEY_STORAGE_ACCESS_FAILED;
        }
        if (keystore_dir_stat.st_mode & S_IRWXO) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Invalid mode (0%o) of key-store directory. Others must not have any permission.",
                       keystore_dir_stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID));
            err = SDC_KEY_STORAGE_ACCESS_FAILED;
        }
    }
    if (err == SDC_OK) {
        if (faccessat(ctx->keystore_dir_fd,
                      ".", (F_OK | R_OK | X_OK), 0)) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Daemon failed to access key-store directory.");
            err = SDC_KEY_STORAGE_ACCESS_FAILED;
        }
    }

    /* verify that daemon can access the key-stores */
    if (err == SDC_OK) {
        if (!faccessat(ctx->keystore_dir_fd,
                       ctx->keystore_names[KEYSTORE_STANDARD],
                       F_OK, 0)) {
            if (faccessat(ctx->keystore_dir_fd,
                          ctx->keystore_names[KEYSTORE_STANDARD],
                          R_OK, 0)) {
                daemon_log(DAEMON_LOG_ERROR,
                           "Daemon failed to access key-store - %s",
                           ctx->keystore_names[KEYSTORE_STANDARD]);
                err = SDC_KEY_STORAGE_ACCESS_FAILED;
            }
            key_store_present = true;
        }
    }
    if (err == SDC_OK) {
        if (!faccessat(ctx->keystore_dir_fd,
                       ctx->keystore_names[KEYSTORE_PENDING],
                       F_OK, 0)) {
            if (faccessat(ctx->keystore_dir_fd,
                          ctx->keystore_names[KEYSTORE_PENDING],
                          R_OK, 0)) {
                daemon_log(DAEMON_LOG_ERROR,
                           "Daemon failed to access key-store - %s",
                           ctx->keystore_names[KEYSTORE_PENDING]);
                err = SDC_KEY_STORAGE_ACCESS_FAILED;
            }
            key_store_present = true;
        }
    }

    if (err == SDC_OK) {
        if (!key_store_present) {
            daemon_log(DAEMON_LOG_ERROR,
                       "No key-store present");
            err = SDC_KEY_STORAGE_ACCESS_FAILED;
        } else {
            /* Generate a warning if the daemon is not able to create or remove
             * the keystore (e.g. in readonly filesystem)
             */
            if (faccessat(ctx->keystore_dir_fd,
                          ".", (W_OK | X_OK), 0)) {
                daemon_log(DAEMON_LOG_WARNING,
                           "WARNING: no write access to key-store folder.\n"
                           "Any attempt to add or remove a persistent key will result in error.");
            }
        }
    }

    return err;
}

/**
 * Modifications are written to pending keystore first.
 * In case anything goes wrong (e.g. key can't be installed to kernel)
 * these modifications are discarded and the pending file is removed
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \param ignore_enoent In case of false no existing pending keystore file results in error
 * \return SDC_KEY_STORAGE_ACCESS_FAILED accessing pending keystore failed
 * \return SDC_OK all good
 */
static sdc_error_t keystore_discard_pending(keystore_ctx_t *ctx, bool ignore_enoent)
{
    int res;

    /* in case not existing file shall be ignored we need to check the existence first.
     * On read only filesystem unlinkat will return ROFS error even if file does not exist
     * */
    if (ignore_enoent) {
        res = faccessat(ctx->keystore_dir_fd,
                        ctx->keystore_names[KEYSTORE_PENDING], F_OK, 0);

        /* return if file does not exist */
        if ((res != 0) && (errno == ENOENT)) {
            return SDC_OK;
        }
    }

    if (unlinkat(ctx->keystore_dir_fd,
                 ctx->keystore_names[KEYSTORE_PENDING],
                 0)) {
        daemon_log(DAEMON_LOG_ERROR, "Failed to discard pending keystore: %s",
                   strerror(errno));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    return SDC_OK;
}


/**
 * Modifications are written to pending keystore first.
 * In case everything works fine (i.e. install to kernel) the original keystore
 * is removed and a hard link to pending keystore is created.
 * Finally the pending keystore file is removed.
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \param initial If true the fact that there is no keystore file won't result in error message
 * \return SDC_KEY_STORAGE_ACCESS_FAILED accessing pending keystore failed
 * \return SDC_OK all good
 */
static sdc_error_t keystore_commit_pending(keystore_ctx_t *ctx, bool initial)
{
    sdc_error_t err = SDC_OK;

    if (!initial) {
        if (fsync(ctx->keystore_dir_fd)) {
            daemon_log(DAEMON_LOG_WARNING, "Failed to sync keystore directory before unlink of keystore: %s",
                       strerror(errno));
        }

        if (unlinkat(ctx->keystore_dir_fd,
                     ctx->keystore_names[KEYSTORE_STANDARD],
                     0)) {
            daemon_log(DAEMON_LOG_WARNING, "Failed to unlink keystore %s: %s",
                       ctx->keystore_names[KEYSTORE_STANDARD],
                       strerror(errno));
        }
    }

    if (linkat(ctx->keystore_dir_fd,
               ctx->keystore_names[KEYSTORE_PENDING],
               ctx->keystore_dir_fd,
               ctx->keystore_names[KEYSTORE_STANDARD],
               0)) {
        daemon_log(DAEMON_LOG_ERROR, "Failed to link pending keystore %s to normal keystore %s: %s",
                   ctx->keystore_names[KEYSTORE_PENDING],
                   ctx->keystore_names[KEYSTORE_STANDARD],
                   strerror(errno));
        err = SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    if (fsync(ctx->keystore_dir_fd)) {
        daemon_log(DAEMON_LOG_ERROR, "Failed to sync keystore directory: %s",
                   strerror(errno));
        err = SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    /* in case linking the keystore or syncing the directory fails
     * we keep the pending keystore - avoid that both keystores are gone.
     * As long as this pending keystore exists installing removing
     * any other key will fail (due to O_CREAT | O_EXCL).
     * In case the root cause does no longer persist the next
     * reboot will remove an additional pending key-store
     */
    if (err == SDC_OK) {
        /* ignore errors */
        (void)keystore_discard_pending(ctx, false);
    }

    return SDC_OK;
}


/**
 * Store persistent keys of ctx to keystore file.
 * Write pending keystore and commit it
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \return SDC_KEY_STORAGE_ACCESS_FAILED accessing pending keystore failed
 * \return SDC_OK all good
 */
static sdc_error_t keystore_save_persistently(keystore_ctx_t *ctx)
{
    int fd;
    sdc_error_t err = SDC_OK;
    size_t stored_keys;

    fd = openat(ctx->keystore_dir_fd, ctx->keystore_names[KEYSTORE_PENDING], O_RDWR | O_CREAT | O_EXCL, NEW_KEYSTORE_MODE);
    if (fd < 0) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Unexpected error when creating pending key-store: %s",
                   strerror(errno));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    err = keyman_store_persistent_keystore(ctx->libkeyman_ctx, fd, &stored_keys);

    close(fd);

    if (err == SDC_OK) {
        daemon_log(DAEMON_LOG_INFO,
                   "%d keys stored to persistent_key-storage.", stored_keys);

        err = keystore_commit_pending(ctx, false);
    } else {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to store key-store persistently: %s",
                   sdc_get_error_string(err));

        (void)keystore_discard_pending(ctx, false);
    }

    return err;
}

sdc_error_t keystore_restore_persistent(keystore_ctx_t *ctx)
{

    int fd;
    sdc_error_t err;
    bool using_pending = false;
    size_t loaded_keys;
    size_t existing_keys;

    fd = openat(ctx->keystore_dir_fd, ctx->keystore_names[KEYSTORE_STANDARD], O_RDONLY);
    if (fd < 0) {
        if (errno == ENOENT) {
            /* try pending key-store */
            fd = openat(ctx->keystore_dir_fd, ctx->keystore_names[KEYSTORE_PENDING], O_RDONLY);
            using_pending = true;
        }
    }

    if (fd < 0) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Unexpected error when accessing key-store: %s",
                   strerror(errno));
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    err = keyman_load_persistent_keystore(ctx->libkeyman_ctx, fd, &loaded_keys, &existing_keys);

    close(fd);

    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to load persistent key-store: %s",
                   sdc_get_error_string(err));
    } else {
        if (existing_keys > 0) {
            daemon_log(DAEMON_LOG_WARNING,
                       "%d keys not loaded as already existing.", existing_keys);
        }
        daemon_log(DAEMON_LOG_INFO,
                   "%d keys loaded from persistent_key-storage.", loaded_keys);

        if (using_pending) {
            err = keystore_commit_pending(ctx, false);
        } else {
            err = keystore_discard_pending(ctx, true);
        }
    }

    return err;
}

sdc_error_t keystore_unlock(keystore_ctx_t *ctx)
{
    sdc_error_t err;

    err = keyman_lock_unlock_keystore(ctx->libkeyman_ctx, LIBKEYMAN_KEYSTORE_ALL, false);
    if (err != SDC_OK) {
        if (err == SDC_NOT_SUPPORTED) {
            daemon_log(DAEMON_LOG_INFO,
                       "Unlocking the key-store not implemented in this architecture");
            err = SDC_OK; /* TODO : remove ignoring missing activate */
        } else {
            daemon_log(DAEMON_LOG_ERROR,
                       "Failed to unlock the key-store key: %s",
                       sdc_get_error_string(err));
        }
    }

    return err;
}

/**
 * Allocate permissions and set gid and uid
 *
 * \param permissions pointer to return permissions struct
 * \param uid user id to set to permissions
 * \param gid group id to set to permissions
 *
 * \return corresponding sdc_error
 */
static sdc_error_t keystore_install_product_key_prepare(
    sdc_permissions_t **permissions, uid_t uid, gid_t gid)
{
    sdc_error_t err;

    err = sdc_permissions_alloc(permissions);
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to allocate permission structure: %s",
                   sdc_get_error_string(err));
    }

    if (err == SDC_OK) {
        err = sdc_set_default_permissions_and_gid(*permissions, gid);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Failed to initialize permission structure: %s",
                       sdc_get_error_string(err));
        }
    }

    if (err == SDC_OK) {
        err = sdc_permissions_set_uid(*permissions, uid);
        if (err != SDC_OK) {
            daemon_log(DAEMON_LOG_ERROR,
                       "Failed to set uid to permission structure: %s",
                       sdc_get_error_string(err));
        }
    }

    return err;
}

/**
 * save keys to persistent keystore and activate afterwards
 *
 * \param ctx       Keystore context
 * \param kid		Key id to activate
 *
 * \return corresponding sdc_error
 */
static sdc_error_t keystore_install_common_save_and_activate (
    keystore_ctx_t *ctx,
    sdc_key_id_t kid
    )
{
    sdc_error_t err = SDC_OK;

    err = keystore_save_persistently(ctx);

    if (err == SDC_OK) {
        err = keyman_activate_key(ctx->libkeyman_ctx, kid);
        if (err != SDC_OK) {
            if (err == SDC_NOT_SUPPORTED) {
                daemon_log(DAEMON_LOG_INFO,
                           "Activate key not implemented in this architecture");
                err = SDC_OK; /* TODO : remove ignoring missing activate */
            } else {
                daemon_log(DAEMON_LOG_ERROR,
                           "Failed to activate key: %s",
                           sdc_get_error_string(err));
            }
        }
    }

    return err;
}

void keystore_overwrite_secret(uint8_t *secret, size_t secretlen)
{
    memset(secret, 0, secretlen);
    asm volatile ("" ::: "memory");
}

/**
 *  Cleanup - free persmissions struct and overwrite key secret
 *
 * \param permissions  permission struct
 * \param key		Key to be overwritten
 * \param err		Will only be overwritten if SDC_OK
 *
 */
static void keystore_install_product_key_cleanup (
    sdc_permissions_t *permissions,
    uint8_t *key_secret, size_t key_secret_len,
    sdc_error_t *err
    )
{
    sdc_error_t tmperr = SDC_OK;

    if (permissions) {
        tmperr = sdc_permissions_free(permissions);
        if (tmperr != SDC_OK) {
            daemon_log(DAEMON_LOG_WARNING,
                       "Failed to free permission structure: %s",
                       sdc_get_error_string(tmperr));
        }
    }

    if (key_secret) {
        keystore_overwrite_secret(key_secret, key_secret_len);
    }

    if (*err == SDC_OK)
        *err = tmperr;
}

sdc_error_t keystore_install_plain_or_random_key(keystore_ctx_t *ctx,
                                                 sdc_key_id_t kid_min,
                                                 sdc_key_id_t kid_max,
                                                 sdc_key_id_t *out_kid,
                                                 const libkeyman_key_spec_t *key_spec,
                                                 bool persistent,
                                                 sdc_keysecret_enc_t key_data_enc,
                                                 const uint8_t *key_data,
                                                 size_t key_data_len)
{
    sdc_error_t err = SDC_OK;

    /* check kid min, max */
    if ((kid_min == SDC_FLAG_INVALID_KID) ||
        (kid_max == SDC_FLAG_INVALID_KID) ||
        (kid_min > kid_max))
        return SDC_KEY_INVALID;

    *out_kid = SDC_FLAG_INVALID_KID;

    if (key_data) {
        if (kid_min != kid_max) {
            err = keyman_generate_plain_storage_key_automatic_kid(
                ctx->libkeyman_ctx,
                kid_min,
                kid_max,
                out_kid,
                key_spec,
                persistent,
                !persistent,
                key_data_enc,
                key_data,
                key_data_len);
        } else {
            err = keyman_generate_plain_storage_key_explicit_kid(
                ctx->libkeyman_ctx,
                kid_min,
                key_spec,
                persistent,
                !persistent,
                key_data_enc,
                key_data,
                key_data_len);
            if (err == SDC_OK)
                *out_kid = kid_min;
        }
    } else {
        if (kid_min != kid_max) {
            err = keyman_generate_random_storage_key_automatic_kid(
                ctx->libkeyman_ctx,
                kid_min,
                kid_max,
                out_kid,
                key_spec,
                persistent,
                !persistent);
        } else {
            *out_kid = kid_min;
            err = keyman_generate_random_storage_key_explicit_kid(
                ctx->libkeyman_ctx,
                kid_min,
                key_spec,
                persistent,
                !persistent);
            if (err == SDC_OK)
                *out_kid = kid_min;
        }
    }
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to install/generate key: %s",
                   sdc_get_error_string(err));
    }

    if ((err == SDC_OK) && (persistent)) {
        err = keystore_install_common_save_and_activate (ctx, *out_kid);
    }

    return err;
}

sdc_error_t keystore_import_key(keystore_ctx_t *ctx,
                                sdc_key_id_t kid_min,
                                sdc_key_id_t kid_max,
                                sdc_key_id_t *out_kid,
                                const libkeyman_key_spec_t *key_spec,
                                bool persistent,
                                sdc_keysecret_enc_t key_data_enc,
                                const uint8_t *key_data,
                                size_t key_data_len)
{
    sdc_error_t err = SDC_OK;

    /* check kid min, max */
    if ((kid_min == SDC_FLAG_INVALID_KID) ||
        (kid_max == SDC_FLAG_INVALID_KID) ||
        (kid_min > kid_max))
        return SDC_KEY_INVALID;

    *out_kid = SDC_FLAG_INVALID_KID;

    if (kid_min != kid_max) {
        err = keyman_import_storage_key_automatic_kid(
            ctx->libkeyman_ctx,
            kid_min,
            kid_max,
            out_kid,
            key_spec,
            persistent,
            !persistent,
            key_data_enc,
            key_data,
            key_data_len);
    } else {
        err = keyman_import_storage_key_explicit_kid(
            ctx->libkeyman_ctx,
            kid_min,
            key_spec,
            persistent,
            !persistent,
            key_data_enc,
            key_data,
            key_data_len);
        if (err == SDC_OK)
            *out_kid = kid_min;
    }
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to import key: %s",
                   sdc_get_error_string(err));
    }

    if ((err == SDC_OK) && (persistent)) {
        err = keystore_install_common_save_and_activate (ctx, *out_kid);
    }

    return err;
}

sdc_error_t keystore_install_product_key(keystore_ctx_t *ctx,

                                         uint8_t *key_secret, size_t key_secret_len,
                                         daemon_credentials_t *cred,
                                         sdc_key_id_t kid)
{
    sdc_error_t err = SDC_OK;
    sdc_permissions_t *permissions = NULL;
    libkeyman_key_spec_t key_spec;

    key_spec.fmt = SDC_KEY_FMT_SIMPLE_SYM;

    if ((key_secret_len == 0) || (!key_secret)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Invalid product key");
        err = SDC_INVALID_PARAMETER;
    }

    err = sdc_key_len_from_bytes(key_secret_len, &(key_spec.len));

    if (err == SDC_OK)
        err = keystore_install_product_key_prepare(
            &permissions, cred->uid, cred->gid);

    if (err == SDC_OK) {
        key_spec.permissions = permissions;

        err = keyman_import_insecure_product_key(
            ctx->libkeyman_ctx, kid,
            &key_spec,
            false,
            SDC_KEY_ENC_PLAIN, key_secret, key_secret_len);
    }

    if (err == SDC_OK) {
        err = keystore_install_common_save_and_activate (ctx, kid);
    }

    keystore_install_product_key_cleanup(
        permissions,
        key_secret, key_secret_len, &err);

    return err;
}


sdc_error_t keystore_remove_key(keystore_ctx_t *ctx, sdc_key_id_t kid,
                                daemon_credentials_t *cred)
{

    sdc_error_t err;
    bool persistent;

    err = keyman_remove_storage_key(ctx->libkeyman_ctx, kid, cred->uid, cred->gid, &persistent);
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Failed to remove key (kid:%d): %s",
                   kid,
                   sdc_get_error_string(err));
    }

    if ((err == SDC_OK) && (persistent)) {
        err = keystore_save_persistently(ctx);
    }

    return err;
}

